<?php

namespace App\Livewire\Admin;

use Livewire\Component;
use Livewire\WithPagination;
use App\Models\ClassSession;
use App\Models\Coach;
use App\Models\Discipline;
use Carbon\Carbon;

class Classes extends Component
{
    use WithPagination;

    protected $paginationTheme = 'tailwind';

    public $search = '';
    public $showModal = false;
    public $editMode = false;

    public $classId;
    public $coach_id;
    public $discipline_id;
    public $start_time;
    public $end_time;
    public $max_capacity;
    public $status = 'disponible';

    public array $statuses = [
        'disponible',
        'llena',
        'cancelada',
        'completada',
    ];

    protected function rules()
    {
        return [
            'coach_id' => 'required|exists:coaches,id',
            'discipline_id' => 'required|exists:disciplines,id',
            'start_time' => 'required|date',
            'end_time' => 'required|date|after:start_time',
            'max_capacity' => 'required|integer|min:1',
            'status' => 'required|in:disponible,llena,cancelada,completada',
        ];
    }

    public function render()
    {
        return view('livewire.admin.classes', [
            'classes' => ClassSession::with(['coach','discipline'])
                ->whereHas('discipline', fn ($q) =>
                    $q->where('name', 'like', "%{$this->search}%")
                )
                ->orderBy('start_time', 'desc')
                ->paginate(10),

            'coaches' => Coach::where('is_active', true)->get(),
            'disciplines' => Discipline::all(),
        ]);
    }

    public function create()
    {
        $this->resetFields();
        $this->editMode = false;
        $this->showModal = true;
    }

    public function edit(ClassSession $class)
    {
        $this->classId = $class->id;
        $this->coach_id = $class->coach_id;
        $this->discipline_id = $class->discipline_id;
        $this->start_time = Carbon::parse($class->start_time)->format('Y-m-d\TH:i');
        $this->end_time = Carbon::parse($class->end_time)->format('Y-m-d\TH:i');
        $this->max_capacity = $class->max_capacity;
        $this->status = $class->status;

        $this->editMode = true;
        $this->showModal = true;
    }

    public function save()
    {
        $this->validate();

        $data = [
            'coach_id' => $this->coach_id,
            'discipline_id' => $this->discipline_id,
            'start_time' => $this->start_time,
            'end_time' => $this->end_time,
            'max_capacity' => $this->max_capacity,
            'status' => $this->status,
        ];

        if ($this->editMode) {
            ClassSession::findOrFail($this->classId)->update($data);
        } else {
            ClassSession::create($data);
        }

        $this->showModal = false;
        $this->resetFields();
    }

    public function setStatus(ClassSession $class, string $status)
    {
        if (!in_array($status, $this->statuses)) {
            return;
        }

        $class->update(['status' => $status]);
    }

    private function resetFields()
    {
        $this->reset([
            'classId',
            'coach_id',
            'discipline_id',
            'start_time',
            'end_time',
            'max_capacity',
            'status',
        ]);
    }
}
