<?php

namespace App\Livewire\Admin;

use Livewire\Component;
use Livewire\WithPagination;
use App\Models\Subscription;
use App\Models\User;
use App\Models\Package;

class Subscriptions extends Component
{
    use WithPagination;

    protected $paginationTheme = 'tailwind';

    public $search = '';
    public $showModal = false;
    public $editMode = false;

    public $subscriptionId;
    public $user_id;
    public $package_id;
    public $status = 'activa';

    public array $statuses = [
        'activa',
        'pausada',
        'cancelada',
        'vencida',
        'pendiente_pago',
    ];

    protected function rules()
    {
        return [
            'user_id' => 'required|exists:users,id',
            'package_id' => 'required|exists:packages,id',
            'status' => 'required|in:activa,pausada,cancelada,vencida,pendiente_pago',
        ];
    }

    public function render()
    {
        return view('livewire.admin.subscriptions', [
            'subscriptions' => Subscription::with(['user','package'])
                ->whereHas('user', fn ($q) =>
                    $q->where('name', 'like', "%{$this->search}%")
                )
                ->orderBy('start_date', 'desc')
                ->paginate(10),

            'users' => User::role('cliente')->get(),
            'packages' => Package::where('is_active', true)->get(),
        ]);
    }

    public function create()
    {
        $this->resetFields();
        $this->editMode = false;
        $this->showModal = true;
    }

    public function edit(Subscription $subscription)
    {
        $this->subscriptionId = $subscription->id;
        $this->user_id = $subscription->user_id;
        $this->package_id = $subscription->package_id;
        $this->status = $subscription->status;

        $this->editMode = true;
        $this->showModal = true;
    }

    public function save()
    {
        $this->validate();

        $package = Package::findOrFail($this->package_id);

        if ($this->editMode) {

            // 🔒 NO tocar fechas ni clases en edición
            Subscription::findOrFail($this->subscriptionId)->update([
                'user_id' => $this->user_id,
                'package_id' => $this->package_id,
                'status' => $this->status,
            ]);

        } else {

            // ✅ CREACIÓN CORRECTA
            $startDate = now();
            $endDate = $startDate->copy()->addDays($package->duration_days);

            Subscription::create([
                'user_id' => $this->user_id,
                'package_id' => $package->id,
                'start_date' => $startDate,
                'end_date' => $endDate,
                'status' => $this->status,

                // 🔥 MAPEO CORRECTO Y ÚNICO
                'remaining_classes' => $package->classes_limit,
            ]);
        }

        $this->showModal = false;
        $this->resetFields();
    }

    private function resetFields()
    {
        $this->reset([
            'subscriptionId',
            'user_id',
            'package_id',
            'status',
        ]);
    }
}
